//www.elegoo.com
//2016.12.09

/*
   --------------------------------------------------------------------------------------------------------------------
   Example to change UID of changeable MIFARE card.
   --------------------------------------------------------------------------------------------------------------------
   This is a MFRC522 library example; for further details and other examples see: https://github.com/miguelbalboa/rfid

   This sample shows how to set the UID on a UID changeable MIFARE card.
   NOTE: for more informations read the README.rst

   @author Tom Clement
   @license Released into the public domain.

   Typical pin layout used:
   -----------------------------------------------------------------------------------------
               MFRC522      Arduino       Arduino   Arduino    Arduino          Arduino
               Reader/PCD   Uno           Mega      Nano v3    Leonardo/Micro   Pro Micro
   Signal      Pin          Pin           Pin       Pin        Pin              Pin
   -----------------------------------------------------------------------------------------
   RST/Reset   RST          9             5         D9         RESET/ICSP-5     RST
   SPI SS      SDA(SS)      10            53        D10        10               10
   SPI MOSI    MOSI         11 / ICSP-4   51        D11        ICSP-4           16
   SPI MISO    MISO         12 / ICSP-1   50        D12        ICSP-1           14
   SPI SCK     SCK          13 / ICSP-3   52        D13        ICSP-3           15
*/

#include <SPI.h>
#include <MFRC522.h>
#include <nRF24L01.h>
#include <RF24.h>
#include <Servo.h>
#include <avr/wdt.h>

const uint32_t RADIO_SIGNATURE[2]={0x12345678, 0xABCDEF};
const byte RADIO_ADDRESS[6] = "00001"; 

#define LED1 12
#define RFID1_RST 28
#define RFID1_SS 29

#define LED2 11
#define RFID2_RST 32
#define RFID2_SS 33

#define LED3 10
#define RFID3_RST 36
#define RFID3_SS 37

#define RADIO_CE 40
#define RADIO_CSN 41

MFRC522 g_rfid1(RFID1_SS, RFID1_RST);   // SS, RST
MFRC522 g_rfid2(RFID2_SS, RFID2_RST);   // SS, RST
MFRC522 g_rfid3(RFID3_SS, RFID3_RST);   // SS, RST

RF24 radio(RADIO_CE, RADIO_CSN); // CE, CSN

struct SkullDetector
{
  int ledPin;
  int rstPin;
  MFRC522 *rfid;
  uint32_t id;
};

SkullDetector g_detectors[3] =
{
  {LED1, RFID1_RST, &g_rfid1},
  {LED2, RFID2_RST, &g_rfid2},
  {LED3, RFID3_RST, &g_rfid3},
};

int g_current = 3;
int g_frame = 0;

void setup()
{
  Serial.begin(9600);  // Initialize serial communications with the PC
  while (!Serial);     // Do nothing if no serial port is opened (added for Arduinos based on ATMEGA32U4)

  Serial.println("setup slave");
  SPI.begin();         // Init SPI bus

  for (int i = 0; i < 3; i++)
  {
    g_detectors[i].rfid->PCD_Init();
    pinMode(g_detectors[i].ledPin, OUTPUT);
    digitalWrite(g_detectors[i].ledPin, LOW);
  }

  radio.begin();
  radio.powerUp();
  radio.openWritingPipe(RADIO_ADDRESS);
  radio.setPALevel(RF24_PA_MAX); 
  radio.setDataRate(RF24_250KBPS);
  radio.setChannel(108);
  radio.enableDynamicAck();
  radio.setAutoAck(false);
  radio.stopListening();
  radio.powerDown();

  wdt_enable(WDTO_1S);
}

void loop()
{
  wdt_reset();
  if (g_current == 3)
  {
    uint32_t ids[5] = {RADIO_SIGNATURE[0], RADIO_SIGNATURE[1], g_detectors[0].id, g_detectors[1].id, g_detectors[2].id};
    Serial.println("radio.write");
    radio.powerUp();
    if (radio.write(&ids, sizeof(ids)))
    {
      Serial.println("success");
    }
    else
    {
      Serial.println("fail");
    }
    radio.powerDown();

    g_frame++;
    Serial.print(g_frame);
    Serial.print(" ");
    Serial.print(ids[2], HEX);
    Serial.print(" ");
    Serial.print(ids[3], HEX);
    Serial.print(" ");
    Serial.println(ids[4], HEX);
  }
  else
  {
    auto &detector = g_detectors[g_current];
    detector.rfid->PCD_Init();
    detector.rfid->PICC_ReadCardSerial();
    detector.rfid->PICC_IsNewCardPresent();
    if (detector.rfid->PICC_ReadCardSerial())
    {
      digitalWrite(detector.ledPin, HIGH);
      detector.id = *reinterpret_cast<uint32_t*>(detector.rfid->uid.uidByte);
    }
    else
    {
      digitalWrite(detector.ledPin, LOW);
      detector.id = 0;
    }
    digitalWrite(detector.rstPin, LOW); // this should power down the sensor until the next PCD_Init
  }

  g_current = (g_current + 1) % 4;
}

