#include <Elegoo_GFX.h>    // Core graphics library
#include <Elegoo_TFTLCD.h> // Hardware-specific library
#include <TouchScreen.h>
#include <avr/pgmspace.h>
#include <nRF24L01.h> 
#include <RF24.h> 
#include <Servo.h>

const uint32_t RADIO_SIGNATURE[2]={0x12345678, 0xABCDEF};
const byte RADIO_ADDRESS[6] = "00001"; 

#define SKULL_WIDTH 90
#define SKULL_HEIGHT 90

#define REAL_SKULLS // undefine this to use solid colors instead of the bitmaps (useful for faster compile and upload during development)

#ifdef REAL_SKULLS
#include "skull_icons.h"
#endif

#define YP A3  // must be an analog pin, use "An" notation!
#define XM A2  // must be an analog pin, use "An" notation!
#define YM 9   // can be a digital pin
#define XP 8   // can be a digital pin

//Touch For New ILI9341 TP
#define TS_MINX 120
#define TS_MAXX 900

#define TS_MINY 70
#define TS_MAXY 920

// For better pressure precision, we need to know the resistance
// between X+ and X- Use any multimeter to read it
// For the one we're using, its 300 ohms across the X plate
TouchScreen ts = TouchScreen(XP, YP, XM, YM, 300);

#define LCD_CS A3
#define LCD_CD A2
#define LCD_WR A1
#define LCD_RD A0
// optional
#define LCD_RESET A4

// Assign human-readable names to some common 16-bit color values:
#define	BLACK   0x0000
#define	BLUE    0x001F
#define	RED     0xF800
#define	GREEN   0x07E0
#define CYAN    0x07FF
#define MAGENTA 0xF81F
#define YELLOW  0xFFE0
#define WHITE   0xFFFF

Elegoo_TFTLCD tft(LCD_CS, LCD_CD, LCD_WR, LCD_RD, LCD_RESET);

const int SCREEN_WIDTH = 320;
const int SCREEN_HEIGHT = 240;
const int FRAMERATE = 20;

RF24 radio(41, 40); // CE, CSN

#define SERVO 33

Servo g_servo;
int g_servoAngle;

void DrawPixels( const uint16_t *src, int x, int y, int w, int h )
{
  if (y + h <= 0) return;
  int start = 0, end = w;
  if (x < 0) start = -x;
  if (x + end > SCREEN_WIDTH) end = SCREEN_WIDTH - x;
  if (end <= start) return;
  if (y + h > SCREEN_HEIGHT) h = SCREEN_HEIGHT - y;
  if (h < 0) return;

  int end2 = end;
  if (end - start > 255) end = start + 255;
  for (int yy = 0; yy < h; yy++
#ifdef REAL_SKULLS
  , src += w
#endif
  )
  {
    if (y + yy >= SCREEN_HEIGHT) break;
    if (y + yy >= 0)
    {
      uint16_t buf[255];
#ifdef REAL_SKULLS
      memcpy_P(buf, src + start, (end - start) * 2);
#else
      for (int i = 0;i < 255; i++)
        buf[i] = *src;
#endif
      tft.setAddrWindow(x + start, y + yy, x + end - 1, y + yy);
      tft.pushColors(buf, end - start, true);
      if (end2 > end)
      {
#ifdef REAL_SKULLS
        memcpy_P(buf, src + end, (end2 - end) * 2);
#endif
        tft.setAddrWindow(x + end, y + yy, x + end2 - 1, y + yy);
        tft.pushColors(buf, end2 - end, true);
      }
    }
  }
}

void RotateServo( int angle )
{
  if (g_servoAngle == angle)
    return;
  g_servo.attach(SERVO);
  const int SERVO_STEPS = angle < g_servoAngle ? 5 : 20;
  for (int i = 0; i <= SERVO_STEPS; i++)
  {
    int a = (g_servoAngle*(SERVO_STEPS-i) + angle*i)/SERVO_STEPS;
    g_servo.write(a);
    delay(50);
    PollRadio();
  }
  g_servoAngle = angle;
  g_servo.detach();
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef REAL_SKULLS
uint16_t g_SkullIcon1[] = {RED};
uint16_t g_SkullIcon2[] = {GREEN};
uint16_t g_SkullIcon3[] = {BLUE};
#endif

static const uint32_t g_SkullIds[3] = {0xCF501455, 0xCF500155, 0xCF5021A5};
static const uint16_t *g_SkullIcons[3]={ g_SkullIcon1, g_SkullIcon2, g_SkullIcon3 };
static int g_CurrentSkulls[3];

static const byte g_Permutations[6][3]={
  {1,2,3},
  {2,1,3},
  {1,3,2},
  {3,1,2},
  {3,2,1},
  {2,3,1},
};

static int g_CurrentPermutation;
static byte g_CurrentState;

static int g_Level;
const int MAX_LEVEL = 5;
const int SERVO_MIN = 30;
const int SERVO_LEVEL = 15;
bool g_buttonPressed;

void PollRadio( void )
{
  if (radio.available()) { 
    uint32_t ids[5]; 
    radio.read(&ids, sizeof(ids)); 
    if (ids[0] == RADIO_SIGNATURE[0] && ids[1] == RADIO_SIGNATURE[1])
    {
      for (int i = 0; i < 3; i++)
      {
        g_CurrentSkulls[i] = 0;
        for (int j = 0; j < 3; j++)
        {
          if (ids[i+2] == g_SkullIds[j])
          {
            g_CurrentSkulls[i] = j+1;
            break;
          }
        }
      }
      Serial.print(g_CurrentSkulls[0]);
      Serial.print(" ");
      Serial.print(g_CurrentSkulls[1]);
      Serial.print(" ");
      Serial.println(g_CurrentSkulls[2]);
    }
  }
}

void DrawPermutation( int permutation )
{
  const byte *perm = g_Permutations[permutation];
  DrawPixels(g_SkullIcons[perm[0]-1],25,10,SKULL_WIDTH,SKULL_HEIGHT);
  DrawPixels(g_SkullIcons[perm[1]-1],115,10,SKULL_WIDTH,SKULL_HEIGHT);
  DrawPixels(g_SkullIcons[perm[2]-1],205,10,SKULL_WIDTH,SKULL_HEIGHT);
}

void DrawState( byte state )
{
  tft.fillCircle(70, 120, 5, (state&0x3) ? WHITE : BLACK);
  tft.fillCircle(160, 120, 5, (state&0xC) ? WHITE : BLACK);
  tft.fillCircle(250, 120, 5, (state&0x30) ? WHITE : BLACK);
}

void DrawSuccess( void )
{
  // Insert your own success message here
  tft.fillScreen(0x1F1F);
  tft.setTextColor(BLACK);
  tft.setTextSize(2);
  tft.setTextWrap(false);
  tft.setCursor(90,80);
  tft.print("Set sail for");
  tft.setCursor(90,100);
  tft.print("Femur Island");
  tft.setCursor(78,120);
  tft.print("(room      x3)");
  tft.fillRect(141,120,16,16,YELLOW);
  tft.fillRect(157,120,16,16,GREEN);
  tft.fillRect(173,120,16,16,RED);
  tft.fillRect(189,120,16,16,BLUE);
}

void SetLevel( int level )
{
  g_Level = level;
  int16_t w = level * 320 / MAX_LEVEL;
  tft.fillRect(0, 200, w, 20, GREEN);
  tft.fillRect(w, 200, 320, 20, 0x1F1F);
  RotateServo(SERVO_MIN + level * SERVO_LEVEL);
}

void SkullSetup( void )
{
  g_buttonPressed = false;
  tft.fillRoundRect(64, 104, 200, 30, 4, BLACK);
  tft.drawRoundRect(63, 103, 200, 30, 4, BLACK);
  tft.drawRoundRect(62, 102, 200, 30, 4, BLACK);
  tft.drawRoundRect(61, 101, 200, 30, 4, BLACK);
  tft.fillRoundRect(60, 100, 200, 30, 4, WHITE);
  tft.drawRoundRect(60, 100, 200, 30, 4, BLACK);
  tft.setTextColor(BLACK);
  tft.setTextSize(2);
  tft.setTextWrap(false);
  tft.setCursor(90,107);
  tft.print("Raise Anchor");
  randomSeed(analogRead(10));
  g_CurrentPermutation = random(0, 6);
  SetLevel(0);
}

void SkullLoop( void )
{
  if (g_Level == MAX_LEVEL) return;
  const byte *perm = g_Permutations[g_CurrentPermutation];
  bool match = true;
  byte nextState = 0;
  int count = 0;
  for (int i = 0; i < 3; i++)
  {
    if (g_CurrentSkulls[i]) count++;
    nextState |= g_CurrentSkulls[i] << (i*2);
    if (perm[i] != g_CurrentSkulls[i])
    {
      match = false;
    }
  }
  if (nextState == g_CurrentState)
  {
    return;
  }

  g_CurrentState = nextState;
  DrawState(g_CurrentState);

  if (match)
  {
    Serial.println("match");
    tft.fillCircle(70, 120, 5, GREEN);
    tft.fillCircle(160, 120, 5, GREEN);
    tft.fillCircle(250, 120, 5, GREEN);
    SetLevel(g_Level + 1);
    for (int i = 0; i < 20; i++)
    {
      PollRadio();
      delay(50);
    }
    if (g_Level == MAX_LEVEL)
    {
      DrawSuccess();
      return;
    }
    DrawState(g_CurrentState);
  }
  else if (count == 3)
  {
    Serial.println("fail");
    SetLevel(0);
  }
  else
  {
    return;
  }

  {
    int next = g_CurrentPermutation;
    while (next == g_CurrentPermutation)
    {
       next = random(0, 6);
    }
    g_CurrentPermutation = next;
    DrawPermutation(g_CurrentPermutation);
  }
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void setup(void)
{
  Serial.begin(9600);
  while (!Serial);     // Do nothing if no serial port is opened (added for Arduinos based on ATMEGA32U4)

  tft.reset();
  tft.begin(0x9341);
  tft.setRotation(3);

  tft.fillScreen(0x1F1F);
  pinMode(13, OUTPUT);
  pinMode(XM, OUTPUT);
  pinMode(YP, OUTPUT);

  radio.begin();
  radio.powerUp();
  radio.openReadingPipe(0, RADIO_ADDRESS);
  radio.setPALevel(RF24_PA_MAX); 
  radio.setDataRate(RF24_250KBPS);
  radio.setChannel(108);
  radio.enableDynamicAck();
  radio.setAutoAck(false);

  radio.startListening(); 

  g_servo.attach(SERVO);
  g_servo.write(SERVO_MIN);
  g_servoAngle = SERVO_MIN;
  delay(300);
  g_servo.detach();

  SkullSetup();
}

void loop()
{
  if (!g_buttonPressed)
  {
    digitalWrite(13, HIGH);
    TSPoint p = ts.getPoint();
    digitalWrite(13, LOW);
    pinMode(XM, OUTPUT);
    pinMode(YP, OUTPUT);
    p.x = map(p.x, TS_MINX, TS_MAXX, tft.width(), 0);
    p.y = (tft.height()-map(p.y, TS_MINY, TS_MAXY, tft.height(), 0));
  
    if (p.z > 10 && p.z < 1000) {
      tft.fillScreen(0x1F1F);
      tft.setTextColor(RED);
      tft.setTextSize(2);
      tft.setTextWrap(false);
      tft.setCursor(10,10);
      tft.print("The anchor is stuck.");
      tft.setCursor(10,30);
      tft.print("Send somebody to the deck");
      tft.setCursor(10,50);
      tft.print("below to operate the");
      tft.setCursor(10,70);
      tft.print("mechanism manually.");
      delay(10000);
      tft.fillScreen(0x1F1F);
      g_buttonPressed = true;
      DrawPermutation(g_CurrentPermutation);
      DrawState(0);
    }
  }

  if (g_buttonPressed)
  {
    PollRadio();
    SkullLoop();
    delay(FRAMERATE);
  }
}

