// KERMITRON 3000

// PINS
#define BUTTON_1 6
#define BUTTON_2 4
#define BUTTON_3 2
#define LED_1 7
#define LED_2 5
#define LED_3 3
#define ARDUINO_TX 8  //connect to RX of the module
#define ARDUINO_RX 9  //should connect to TX of the Serial MP3 Player module
#define LED_DATA_PIN 10
#define ARM_PIN 11

// STATE
static int g_Song = 0;
static int g_SongSelection = 0;
static bool g_bAnimate = true;
static bool g_bLights = true;
static long g_SongDuration[5] = {0, 185, 219, 222, 21};
//static long g_SongDuration[5] = {0, 10, 30, 30, 21};
static long g_SongBeat[5] = {0, 1560, 1300, 1666, 500};
static long g_SongBeatOffset[5] = {0, 440, 0, 1000, 0};
static long g_SongStart = 0;

#include <SoftwareSerial.h>
#include <FastLED.h>
#include <EEPROM.h>

// LED globals
FASTLED_USING_NAMESPACE

#define LED_TYPE    WS2812B
#define COLOR_ORDER GRB
#define NUM_LEDS    40
CRGB leds_L[NUM_LEDS];
CRGB leds_P[NUM_LEDS];

#define BRIGHTNESS          10
#define FRAMES_PER_SECOND  120

// MP3 globals
#define PRINT_MP3_ANSWER 0
SoftwareSerial mp3(ARDUINO_RX, ARDUINO_TX);

static int8_t Send_buf[8] = {0}; // Buffer for Send commands.  // BETTER LOCALLY
static uint8_t ansbuf[10] = {0}; // Buffer for the answers.    // BETTER LOCALLY

/************ Command byte **************************/
#define CMD_NEXT_SONG     0X01  // Play next song.
#define CMD_PREV_SONG     0X02  // Play previous song.
#define CMD_PLAY_W_INDEX  0X03
#define CMD_VOLUME_UP     0X04
#define CMD_VOLUME_DOWN   0X05
#define CMD_SET_VOLUME    0X06

#define CMD_SNG_CYCL_PLAY 0X08  // Single Cycle Play.
#define CMD_SEL_DEV       0X09
#define CMD_SLEEP_MODE    0X0A
#define CMD_WAKE_UP       0X0B
#define CMD_RESET         0X0C
#define CMD_PLAY          0X0D
#define CMD_PAUSE         0X0E
#define CMD_PLAY_FOLDER_FILE 0X0F

#define CMD_STOP_PLAY     0X16  // Stop playing continuously. 
#define CMD_FOLDER_CYCLE  0X17
#define CMD_SHUFFLE_PLAY  0x18 //
#define CMD_SET_SNGL_CYCL 0X19 // Set single cycle.

#define CMD_SET_DAC 0X1A
#define DAC_ON  0X00
#define DAC_OFF 0X01

#define CMD_PLAY_W_VOL    0X22
#define CMD_PLAYING_N     0x4C
#define CMD_QUERY_STATUS      0x42
#define CMD_QUERY_VOLUME      0x43
#define CMD_QUERY_FLDR_TRACKS 0x4e
#define CMD_QUERY_TOT_TRACKS  0x48
#define CMD_QUERY_FLDR_COUNT  0x4f

/************ Opitons **************************/
#define DEV_TF            0X02

/*********************************************************************/

static int ReadButtons()
{
  int buttons = 0;
  if (digitalRead(BUTTON_1) == LOW)
  {
    buttons |= 1;
  }
  if (digitalRead(BUTTON_2) == LOW)
  {
    buttons |= 2;
  }
  if (digitalRead(BUTTON_3) == LOW)
  {
    buttons |= 4;
  }
  return buttons;
}

static void UpdateLeds()
{
  digitalWrite(LED_1, (g_Song == 1 || g_Song == 4) ? HIGH : LOW);
  digitalWrite(LED_2, (g_Song == 2 || g_Song == 4) ? HIGH : LOW);
  digitalWrite(LED_3, (g_Song == 3 || g_Song == 4) ? HIGH : LOW);
}

static void StartSong( int song )
{
  g_Song = song;
  sendMp3Command(CMD_PLAY_W_INDEX, 0, g_Song);
  UpdateLeds();
  digitalWrite(ARM_PIN, g_bAnimate ? HIGH : LOW);
  Serial.println("start song");
  g_SongStart = millis();
}

static void StopSong( void )
{
  g_Song = 0;
  sendMp3Command(CMD_STOP_PLAY);
  UpdateLeds();
  digitalWrite(ARM_PIN, LOW);
  memset(leds_L, 0, sizeof(leds_L));
  Serial.println("stop song");
  g_SongStart = 0;
}

static void CopyLeds( void )
{
  for (int row = 0; row < 8; row++)
    for (int col = 0; col < 5; col++)
    {
      int l = col * 8 + row;
      int p = row * 5 + ((row&1)?(4-col):col);
      leds_P[p] = leds_L[l];
    }
}

static void ReadSettings( void )
{
  int rom = EEPROM[0];
//  rom = 0x31; // 00110001
  g_SongSelection = (rom&15);
  if (g_SongSelection < 1 || g_SongSelection > 4) g_SongSelection = 4;
  g_bAnimate = (rom&16) != 0;
  g_bLights = (rom&32) != 0;
}

static void WriteSettings( void )
{
  int rom = (g_SongSelection ? (g_SongSelection&15) : 1) | (g_bAnimate ? 16 : 0) | (g_bLights ? 32 : 0);
  EEPROM.write(0, rom);
}

void setup()
{
  Serial.begin(9600);
  mp3.begin(9600);
  pinMode(BUTTON_1, INPUT_PULLUP);
  pinMode(LED_1, OUTPUT);
  pinMode(BUTTON_2, INPUT_PULLUP);
  pinMode(LED_2, OUTPUT);
  pinMode(BUTTON_3, INPUT_PULLUP);
  pinMode(LED_3, OUTPUT);
  pinMode(ARM_PIN, OUTPUT);
  digitalWrite(ARM_PIN, LOW);
  sendMp3Command(CMD_SEL_DEV, 0, DEV_TF);
  delay(500);
  
  sendMp3Command(CMD_SET_VOLUME, 0, 25);

  // tell FastLED about the LED strip configuration
  FastLED.addLeds<LED_TYPE,LED_DATA_PIN,COLOR_ORDER>(leds_P, NUM_LEDS).setCorrection(TypicalLEDStrip);
  FastLED.setBrightness(BRIGHTNESS);
  FastLED.clear(true);

  ReadSettings();
  StartSong(g_SongSelection);
}

static CRGB g_Rainbow[5] = {
  CRGB(139, 0, 255),
  CRGB(0, 0, 255),
  CRGB(0, 255, 0),
  CRGB(255, 192, 0),
  CRGB(255, 0, 0),
};

void addGlitter( fract8 chanceOfGlitter) 
{
  if( random8() < chanceOfGlitter) {
    leds_L[ random16(NUM_LEDS) ] += CRGB::White;
  }
}

void AnimateRainbowConnection( long playTime )
{
  if (playTime < 1500)
  {
    // fade in
    for (int row = 0; row < 5; row++)
    {
      long dist=abs(playTime-row*300)/3;
      long fade=(255l*150-dist*255)/150;
      if (fade<0) fade=0;
      for (int col = 0; col < 8; col++)
        leds_L[row*8+col] = g_Rainbow[row]%(uint8_t)fade;
    }
  }
  else if (playTime < 7500)
  {
    // intro
    int progress = ((playTime-1500)%3100)/70;
    int pass = (playTime-1500)/3100;
    for (int row = 0; row < 5; row++)
      for (int col = 0; col < 8; col++)
      {
        int fade = (row*8+7-col<progress)?(pass?255:64):(pass?64:0);
        leds_L[row*8+col] = g_Rainbow[row]%(uint8_t)fade;
      }
  }
  else if (playTime < 45500)
  {
    // verse 1
    int roll = ((playTime-7500)%2000);
    int rolli = 4-roll/400;
    int rollf = ((roll%400)*16)/25;
    for (int row = 0; row < 5; row++)
    {
      CRGB c = g_Rainbow[(rolli+row+1)%5].lerp8(g_Rainbow[(rolli+row)%5], rollf);
      for (int col = 0; col < 8; col++)
        leds_L[row*8+col] = c;
    }
  }
  else if (playTime < 57680)
  {
    // chorus 1
    for (int row = 0; row < 5; row++)
    {
      for (int col = 0; col < 8; col++)
      {
        float dist = sqrtf((row-2)*(row-2)+(col-3.5f)*(col-3.5f)*0.25f)/2;
        float phase = dist-((playTime-43300)%1560)*6.28f/1560;
        long fade=(long)((sinf(phase)+0.5)*200);
        if (fade<0) fade=0;
        if (fade>255) fade=255;
        if (playTime > 57000 && fade<60)
          fade=60;
        leds_L[row*8+col] = g_Rainbow[row]%(uint8_t)fade;
      }
    }
  }
  else if (playTime < 99000)
  {
    // verse 2
    float phase0 = ((playTime-1100)%3120)/3120.f;
    for (int col = 0; col < 8; col++)
    {
      float phase = phase0*4+col/4.f;
      float c=cos(phase*1.57f);
      if (c<0) c=0;
      int fade = (int)((c+0.3f)*255/1.3f);
      for (int row = 0; row < 5; row++)
        leds_L[row*8+col] = g_Rainbow[row]%(uint8_t)fade;
    }
  }
  else if (playTime < 109700)
  {
    // chorus 2
    for (int row = 0; row < 5; row++)
    {
      for (int col = 0; col < 8; col++)
      {
        float dist = sqrtf((row-2)*(row-2)+(col-3.5f)*(col-3.5f)*0.25f)/2;
        float phase = dist-((playTime-94970)%1560)*6.28f/1560;
        long fade=(long)(sinf(phase)*255);
        if (fade<0) fade=0;
        leds_L[row*8+col] = g_Rainbow[row]%(uint8_t)fade;
      }
    }
  }
  else if (playTime < 122300)
  {
    // magic
    for (int row = 0; row < 5; row++)
      for (int col = 0; col < 8; col++)
        leds_L[row*8+col] = g_Rainbow[row]%(uint8_t)200;
    addGlitter(64);
  }
  else if (playTime < 160000)
  {
    // verse 3
    int roll = ((playTime-122300)%2000);
    int rolli = 4-roll/400;
    int rollf = ((roll%400)*16)/25;
    for (int row = 0; row < 5; row++)
    {
      CRGB c = g_Rainbow[(rolli+row+1)%5].lerp8(g_Rainbow[(rolli+row)%5], rollf);
      for (int col = 0; col < 8; col++)
        leds_L[row*8+col] = c;
    }
  }
  else if (playTime < 172000)
  {
    // chorus 3
    for (int row = 0; row < 5; row++)
    {
      for (int col = 0; col < 8; col++)
      {
        float dist = sqrtf((row-2)*(row-2)+(col-3.5f)*(col-3.5f)*0.25f)/2;
        float phase = dist-((playTime-95070+900)%1560)*6.28f/1560;
        long fade=(long)(sinf(phase)*255);
        if (fade<0) fade=0;
        leds_L[row*8+col] = g_Rainbow[row]%(uint8_t)fade;
      }
    }
  }
  else
  {
    // lalala
    int progress = ((185000-playTime)%6500)/162;
    int pass = (185000-playTime)/6500;
    for (int row = 0; row < 5; row++)
      for (int col = 0; col < 8; col++)
      {
        int fade = (row*8+7-col<progress)?(pass?255:64):(pass?64:0);
        leds_L[row*8+col] = g_Rainbow[row]%(uint8_t)fade;
      }
  }
}

void AnimateOverTheRainbow( long playTime )
{
  if (playTime < 1500)
  {
    // fade in
    for (int row = 0; row < 5; row++)
    {
      long dist=abs(playTime-row*300)/3;
      long fade=(255l*150-dist*255)/150;
      if (fade<0) fade=0;
      for (int col = 0; col < 8; col++)
        leds_L[row*8+col] = g_Rainbow[row]%(uint8_t)fade;
    }
  }
  else if (playTime < 2800)
  {
    memset(leds_L, 0, sizeof(leds_L));
  }
  else if (playTime < 54000)
  {
    // part 1 verse 1
    float phase0 = ((playTime-2300)%1666)/1666.f;
    for (int col = 0; col < 8; col++)
    {
      float phase = phase0*4+col/4.f;
      float c=cos(phase*1.57f);
      if (c<0) c=0;
      int fade = (int)(c*255);
      for (int row = 0; row < 5; row++)
        leds_L[row*8+col] = g_Rainbow[row]%(uint8_t)fade;
    }
  }
  else if (playTime < 81000)
  {
    // part 1 verse 2
    int roll = (playTime%1665);
    int rolli = 4-roll/333;
    int rollf = ((roll%333)*256l)/333;
    for (int row = 0; row < 5; row++)
    {
      CRGB c = g_Rainbow[(rolli+row+1)%5].lerp8(g_Rainbow[(rolli+row)%5], rollf);
      for (int col = 0; col < 8; col++)
        leds_L[row*8+col] = c;
    }
  }
  else if (playTime < 105200)
  {
    // part 1 verse 3
    float phase0 = ((playTime-80600)%1666)/1666.f;
    for (int col = 0; col < 8; col++)
    {
      float phase = phase0*4+col/4.f;
      float c=cos(phase*1.57f);
      if (c<0) c=0;
      int fade = (int)(c*255);
      for (int row = 0; row < 5; row++)
        leds_L[row*8+col] = g_Rainbow[row]%(uint8_t)fade;
    }
  }
  else if (playTime < 157500)
  {
    // part 2 verse 1
    for (int row = 0; row < 5; row++)
    {
      for (int col = 0; col < 8; col++)
      {
        float dist = sqrtf((row-2)*(row-2)+(col-3.5f)*(col-3.5f)*0.25f)/2;
        float phase = dist-((playTime-104050)%1500)*6.28f/1500;
        long fade=(long)(sinf(phase)*255);
        if (fade<0) fade=0;
        leds_L[row*8+col] = g_Rainbow[row]%(uint8_t)fade;
      }
    }
  }
  else if (playTime < 181000)
  {
    // part 2 verse 2
    for (int row = 0; row < 5; row++)
      for (int col = 0; col < 8; col++)
        leds_L[row*8+col] = g_Rainbow[row]%(uint8_t)200;
    addGlitter(64);
  }
  else if (playTime < 214300)
  {
    // part 2 verse 3
    for (int row = 0; row < 5; row++)
    {
      for (int col = 0; col < 8; col++)
      {
        float dist = sqrtf((row-2)*(row-2)+(col-3.5f)*(col-3.5f)*0.25f)/2;
        float phase = dist-((playTime-213000)%1500)*6.28f/1500;
        long fade=(long)(sinf(phase)*255);
        if (fade<0) fade=0;
        leds_L[row*8+col] = g_Rainbow[row]%(uint8_t)fade;
      }
    }
  }
  else if (playTime < 218700)
  {
    // outro
    int progress = (218300-playTime)/100;
    for (int row = 0; row < 5; row++)
      for (int col = 0; col < 8; col++)
        if ((7-col)*5+row<progress)
          leds_L[row*8+col] = g_Rainbow[row];
        else
          leds_L[row*8+col] = CRGB::Black;
  }
  else if (playTime < 218850)
  {
    for (int row = 0; row < 5; row++)
    {
      long dist=abs((playTime-218700)-row*30)/3;
      long fade=(255l*15-dist*255)/15;
      if (fade<0) fade=0;
      for (int col = 0; col < 8; col++)
        leds_L[row*8+col] = g_Rainbow[row]%(uint8_t)fade;
    }
  }
  else
  {
    memset(leds_L, 0, sizeof(leds_L));
  }
}

static unsigned char g_HappyBirthday[5][8] = {
  {0x00, 0xAA, 0x88, 0xC3, 0x2A, 0x4A, 0xCA, 0xC8},
  {0x00, 0xAA, 0x88, 0x22, 0xAA, 0x4A, 0xAA, 0x20},
  {0x00, 0xEE, 0xCC, 0x63, 0x2C, 0x4E, 0xAE, 0x68},
  {0x00, 0xAA, 0xAA, 0xA2, 0xAA, 0x4A, 0xAA, 0xA8},
  {0x00, 0xA4, 0xCC, 0xA3, 0x2C, 0xEA, 0xC4, 0xA8},
};

void AnimateHappyBirthday( long playTime )
{
  int offset = playTime / 330;
  for (int row = 0; row < 5; row++)
    for (int col = 0; col < 8; col++)
    {
      int i = col + offset;
      if (i > 128) i = 128;
      i &= 63;
      unsigned char b = g_HappyBirthday[row][i/8];
      if (b&(1<<(7-(i&7))))
        leds_L[row*8+7-col] = CRGB(128,255,128);
      else
        leds_L[row*8+7-col] = CRGB::Black;
    }
}

void loop()
{
  if (g_Song == 1)
  {
    AnimateRainbowConnection(millis() - g_SongStart);
  }
  else if (g_Song == 2)
  {
    AnimateRainbowConnection((millis() - g_SongStart)*g_SongDuration[1]/g_SongDuration[2]);
  }
  else if (g_Song == 3)
  {
    AnimateOverTheRainbow(millis() - g_SongStart);
  }
  else if (g_Song == 4)
  {
    AnimateHappyBirthday(millis() - g_SongStart);
  }

  if (g_bLights)
  {
    CopyLeds();
    // send the 'leds' array out to the actual LED strip
    FastLED.show();
    // insert a delay to keep the framerate modest
    FastLED.delay(1000/FRAMES_PER_SECOND); 
}
  else
  {
    FastLED.clear(true);
  }

  int buttons = ReadButtons();
  if (buttons)
  {
    for (int i = 0; i < 100; i++)
    {
      int b = ReadButtons();
      if (!b) break;
      buttons |= b;
      delay(10);
    }
    int song = g_Song;
    bool bAnimate = g_bAnimate;
    bool bLights = g_bLights;
    if (buttons == 1)
      song = 1;
     else if (buttons == 2)
      song = 2;
    else if (buttons == 4)
      song = 3;
    else if (buttons == 3)
      bAnimate = !bAnimate;
    else if (buttons == 6)
      bLights = !bLights;
    else if (buttons == 7)
    {
      song = 4;
      bAnimate = true;
      bLights = true;
    }
    if (song != g_Song)
    {
      StartSong(song);
      g_SongSelection = song;
    }
    if (bAnimate != g_bAnimate)
    {
      g_bAnimate = bAnimate;
      if (!bAnimate)
        digitalWrite(ARM_PIN, LOW);
    }
    if (bLights != g_bLights)
    {
      g_bLights = bLights;
      if (!g_bLights)
      {
        FastLED.clear(true);
      }
    }
    WriteSettings();
  }

#if PRINT_MP3_ANSWER
  memset(ansbuf,0,sizeof(ansbuf));
  if (mp3.available()) 
  { 
   Serial.println(decodeMP3Answer()); 
  } 
#endif

  if (g_Song)
  {
    long playTime = millis() - g_SongStart;
    if (g_bAnimate)
    {
      bool on = (playTime>=g_SongBeatOffset[g_Song]) && ((playTime-g_SongBeatOffset[g_Song])%g_SongBeat[g_Song])<500;
      digitalWrite(ARM_PIN, on ? HIGH : LOW);
    }
    // Check for end of song
    if (playTime > g_SongDuration[g_Song]*1000)
    {
      StopSong();
      if (g_SongSelection == 4)
      {
          delay(1000);
          g_SongSelection = 1;
          WriteSettings();
          StartSong(1);
      }
    }
  }
}

/********************************************************************************/
/*Function: Send command to the MP3                                             */
/*Parameter: byte command                                                       */
/*Parameter: byte dat1 parameter for the command                                */
/*Parameter: byte dat2 parameter for the command                                */
void sendMp3Command(byte command){
  sendMp3Command(command, 0, 0);
}

void sendMp3Command(byte command, byte dat1, byte dat2)
{
  Send_buf[0] = 0x7E;    //
  Send_buf[1] = 0xFF;    //
  Send_buf[2] = 0x06;    // Len
  Send_buf[3] = command; //
  Send_buf[4] = 0x01;    // 0x00 NO, 0x01 feedback
  Send_buf[5] = dat1;    // datah
  Send_buf[6] = dat2;    // datal
  Send_buf[7] = 0xEF;    //
  for (int pass = 0; pass < 3; pass++) // try 3 times to be sure
  {
    delay(20);
    for (uint8_t i = 0; i < 8; i++)
    {
      mp3.write(Send_buf[i]);
    }
  }
}


#if PRINT_MP3_ANSWER
String decodeMP3Answer() { 
  String decodedMP3Answer = ""; 

  decodedMP3Answer += sanswer(); 

  switch (ansbuf[3]) { 
    case 0x3A: 
      decodedMP3Answer += " -> Memory card inserted."; 
      break; 

    case 0x3D: 
      decodedMP3Answer += " -> Completed play num " + String(ansbuf[6], DEC); 
      break; 

    case 0x40: 
      decodedMP3Answer += " -> Error"; 
      break; 

    case 0x41: 
      decodedMP3Answer += " -> Data recived correctly. "; 
      break; 

    case 0x42: 
      decodedMP3Answer += " -> Status playing: " + String(ansbuf[6], DEC); 
      break; 

    case 0x48: 
      decodedMP3Answer += " -> File count: " + String(ansbuf[6], DEC); 
      break; 

    case 0x4C: 
      decodedMP3Answer += " -> Playing: " + String(ansbuf[6], DEC); 
      break; 

    case 0x4E: 
      decodedMP3Answer += " -> Folder file count: " + String(ansbuf[6], DEC); 
      break; 

    case 0x4F: 
      decodedMP3Answer += " -> Folder count: " + String(ansbuf[6], DEC); 
      break; 
  } 

  return decodedMP3Answer; 
} 

String sbyte2hex(uint8_t b) 
{ 
  String shex; 

  shex = "0X"; 

  if (b < 16) shex += "0"; 
  shex += String(b, HEX); 
  shex += " "; 
  return shex; 
} 

String sanswer(void) 
{ 
  uint8_t i = 0; 
  String mp3answer = ""; 

  // Get only 10 Bytes 
  while (mp3.available() && (i < 10)) 
  { 
    uint8_t b = mp3.read(); 
    ansbuf[i] = b; 
    i++; 

    mp3answer += sbyte2hex(b); 
  } 

  // if the answer format is correct. 
  if ((ansbuf[0] == 0x7E) && (ansbuf[9] == 0xEF)) 
  { 
    return mp3answer; 
  } 

  return "???: " + mp3answer; 
} 
#endif

